# KissTelegram Suite

[![License: MIT](https://img.shields.io/badge/License-MIT-yellow.svg)](https://opensource.org/licenses/MIT)
[![Platform: ESP32](https://img.shields.io/badge/Platform-ESP32-blue.svg)](https://www.espressif.com/)
[![Version: 1.0.0](https://img.shields.io/badge/Version-1.0.0-green.svg)](https://github.com/victek/KissTelegram)

**Librería Telegram para IoT con Persistencia, OTA y Power Management**

Una suite completa diseñada para proyectos IoT donde la **pérdida de datos no es aceptable**.

---

## Características

| Característica |           Descripción  |
|---------------------------|-------------|
| **Persistencia LittleFS** | Mensajes almacenados en flash, sobreviven reinicios |
| **Cola con Prioridades**  | CRITICAL > HIGH > NORMAL > LOW                      |
| **Conexión Keep-Alive**   | Reutiliza SSL, reduce latencia 10x                  |
| **OTA Seguro**            | PIN/PUK, backup, rollback automático                |
| **Power Management**      | Modos ACTIVE/IDLE/LOW para batería                  |
| **Rate-Limit Handling**   | Manejo automático del 429 de Telegram               |

---

## Comparativa Rápida

| Prestaciones    | UniversalTelegramBot    | CTBot      | **KissTelegram** |
|-----------------|:-----------------------:|:----------:|:----------------:|
| Persistencia    | ❌                     | ❌         |  ✅              |
| Cola mensajes   | ❌                     | RAM         | ✅ Flash        |
| Keep-alive SSL  | ❌                     | ❌         | ✅ KissSSL      |
| OTA integrado   | ❌                     | ❌         | ✅ KissOTA      |
| Power modes     | ❌                     | ❌         | ✅ KissTelegram |
| Rate-limit      | ❌                     | ❌         | ✅ KissTelegram |
| arduinoJson     | ✅ Fugas-Crash         | ✅ Fugas   | ✅ Parser propio|
---

## Instalación

1. Clona el repositorio en tu carpeta de librerías Arduino:
```bash
cd ~/Arduino/libraries
git clone https://github.com/victek/KissTelegram.git
```

2. O descarga los 2 ZIP,  Suite_kiss_1.zip extrae en `Arduino/libraries/`
   Suite_kiss2.zip, incluye documentación, sketch .ino de test y Partitions.csv

---
## Espacio ocupado en RAM por la libreria --> 55KB
Porqué? Porque es inevitable si vas a usar SSL seguro.
## Espacio ocupado por Kisstelegram con OTA sin certificados y mbedTLS ---> 7-8KB

## Uso Básico

```cpp
#include "KissTelegram.h"
#include "KissCredentials.h"

KissCredentials credentials;
KissTelegram* bot;

void handleMessage(const char* chat_id, const char* text,
                   const char* command, const char* param) {
  if (strcmp(command, "/start") == 0) {
    bot->sendMessage(chat_id, "Hola! Soy KissTelegram");
  }
  else if (strcmp(command, "/status") == 0) {
    char status[128];
    snprintf(status, sizeof(status),
             "Heap: %d bytes\nPendientes: %d",
             ESP.getFreeHeap(), bot->countPendingMessages());
    bot->sendMessage(chat_id, status);
  }
}

void setup() {
  Serial.begin(115200);

  // Conectar WiFi
  WiFi.begin(credentials.getWifiSSID(), credentials.getWifiPassword());
  while (WiFi.status() != WL_CONNECTED) {
    delay(500);
    Serial.print(".");
  }
  Serial.println("\nWiFi conectado!");

  // Inicializar bot
  bot = new KissTelegram(credentials.getBotToken());
  bot->setWifiStable();
  bot->enable();

  // Mensaje de inicio (se encola si no hay conexión)
  bot->queueMessage(credentials.getChatId(),
                    "Sistema iniciado!",
                    KissTelegram::PRIORITY_HIGH);
}

void loop() {
  // Procesar mensajes entrantes
  bot->checkMessages(handleMessage);

  // Enviar mensajes pendientes
  bot->processQueue();

  // Tu código de sensores aquí...
  static unsigned long lastSensor = 0;
  if (millis() - lastSensor > 60000) {  // Cada minuto
    char msg[64];
    snprintf(msg, sizeof(msg), "Temperatura: %.1f°C", readTemperature());
    bot->queueMessage(credentials.getChatId(), msg, KissTelegram::PRIORITY_NORMAL);
    lastSensor = millis();
  }
}
```

---

## Prioridades de Mensajes

```cpp
// Alarmas - se envían primero
bot->queueMessage(chat_id, "🚨 ALARMA!", KissTelegram::PRIORITY_CRITICAL);

// Notificaciones importantes
bot->queueMessage(chat_id, "⚠️ Batería baja", KissTelegram::PRIORITY_HIGH);

// Datos normales
bot->queueMessage(chat_id, "Temp: 25°C", KissTelegram::PRIORITY_NORMAL);

// Logs/Debug - cuando no hay nada más
bot->queueMessage(chat_id, "Heartbeat", KissTelegram::PRIORITY_LOW);
```

---

## OTA (Actualización Remota)

```cpp
#include "KissOTA.h"

KissOTA* ota;

void setup() {
  // ... setup bot ...
  ota = new KissOTA(bot, &credentials);
}

void loop() {
  bot->checkMessages(handleMessage);
  bot->processQueue();

  if (ota->isActive()) {
    ota->loop();
  }
}
```

### Comandos Telegram para OTA:
- `/ota` - Iniciar proceso OTA
- `/otapin XXXX` - Autenticar con PIN
- `/otaok` - Confirmar flash del firmware
- `/otacancel` - Cancelar proceso

### Seguridad OTA:
- Autenticación PIN/PUK
- Backup automático del firmware actual
- Rollback automático si el nuevo firmware falla
- Detección de boot-loop

---

## Arquitectura

```
┌─────────────────────────────────────────────┐
│              Tu Aplicación                  │
└─────────────────────┬───────────────────────┘
                      │
┌─────────────────────▼───────────────────────┐
│            KissTelegram (Core)              │
│  • Envío/Recepción mensajes                 │
│  • Cola con prioridades                     │
│  • Persistencia LittleFS                    │
└──┬─────────┬─────────┬─────────┬───────────┘
   │         │         │         │
┌──▼──┐ ┌────▼───┐ ┌───▼──┐ ┌───▼────┐
│SSL  │ │  OTA   │ │ Time │ │ Config │
└─────┘ └────────┘ └──────┘ └────────┘
```

---

## Requisitos

| Requisito    | Mínimo | Recomendado |
|--------------|--------|-------------|
| Plataforma   | ESP32  | ESP32-S3    |
| Flash        | 4MB    | 8MB+        |
| PSRAM        | -      | 2MB+ (OTA)  |
| RAM libre    | 50KB   | 100KB+      |
| Arduino Core | 2.0.0  | 2.0.14+     |

Mejor dispositivo ESP32S3 N16R8
Partitions.csv para las particiones personalizadas adjunto

---

## Documentación

- [Manual de uso](Documentation/Manual_de_uso.md/User's Manual) Español / Inglés

---

## Estructura de Archivos

```
KissTelegram/
├── KissTelegram.h/cpp      # Core: mensajes, cola
├── KissSSL.h/cpp           # Conexión modo seguro SSL keep-alive
├── KissOTA.h/cpp           # Actualización remota
├── KissTime.h/cpp          # Sincronización NTP
├── KissConfig.h/cpp        # Configuración persistente
├── KissCredentials.h/cpp   # Credenciales NVS
├── system_setup_template.h # Plantilla en blanco. Una vez entrados los datos y passwords renombrar como system_setup.h
├── kiss_suite.ino          # Ejemplos y Pruebas de test
├── Partitions.csv          # Tabla de particiones para ESP32S3 N16R8
├── Documentation/          # Documentación
```

---

## Migración desde otras librerías

### Desde UniversalTelegramBot

```cpp
// ANTES
bot.sendMessage(chat_id, "texto", "");

// DESPUÉS
bot->queueMessage(chat_id, "texto", KissTelegram::PRIORITY_NORMAL);
```

Ver [guía completa de migración](Documentation/Manual de usuario / User's manual).

---

## Licencia

MIT License - Uso libre para proyectos personales y comerciales.

---

## Autor

**Vicente Soriano**
- Email: victek@gmail.com

---

## Contribuir

1. Fork el repositorio
2. Crea tu branch (`git checkout -b feature/nueva-funcionalidad`)
3. Commit tus cambios (`git commit -am 'Añade nueva funcionalidad'`)
4. Push al branch (`git push origin feature/nueva-funcionalidad`)
5. Abre un Pull Request
