#!/bin/bash
#
# build_tools.sh - Do a build of the MIPS toolchain & glibc
#
# Copyright 2001,2003 Broadcom Corporation
#
#

usage ()
{
cat <<EOF
    Usage: $script [-d base_dir] [-k kernel_dir] [-i install_base]
    -d base_dir
              The base directory where the source will be extracted to and
              the release will be built.  If no -d option is provided,
              the default base_dir is the current pwd.

    -k kernel_dir
              The base directory where a copy of the kernel sources,
	      already configured; can be found. If no -k option is
	      provided, the default kernel_dir is
	      $base_dir/../release/src/linux/linux.

    -i install_base
              The base directory where the resulting binaries will be
	      installed and run from.  If no -i option is provided, the
              default install_base is /opt/brcm.

EOF
    exit 1;
}

BUILD_DIR=`pwd`
LINUX=${BUILD_DIR}/../release/src/linux/linux
INSTALL_BASE="/opt/brcm"

while getopts 'd:i:k:h' OPT
do
    case "$OPT" in
    d)
	BUILD_DIR=$OPTARG;
	;;
    i)
	INSTALL_BASE=$OPTARG;
	;;
    k)
	LINUX=$OPTARG;
	;;
    h)
	usage;
	;;
    :)
	echo "Missing required argument for option \"$OPTARG\".\n";
	usage;
	;;
    ?)
	echo "Unrecognized option - \"$OPTARG\".\n";
	usage;
	;;
    esac
done

if [ "$OPTIND" -le "$#" ]; then
    usage;
fi

# Figure out install directory
INSTALL_DIR=${INSTALL_BASE}/hndtools-mipsel-linux-3.0

# Enter build directory
cd "${BUILD_DIR}"
if [ $? != 0 ]; then
    echo Cannot cd to ${BUILD_DIR}
    exit 1
fi;

echo "BUILD_DIR=$BUILD_DIR"
echo "LINUX_DIR=$LINUX"
echo "INSTALL_DIR=$INSTALL_DIR"

echo -n "Building gnutools: "
date

GNU=${BUILD_DIR}/gnu-20010422

cd ${GNU}

mkdir mipsel-nl
cd mipsel-nl

echo -n "Doing gnu-tools with no-libs: "
date

../configure -v --with-newlib --disable-shared --enable-languages=c \
	--prefix=${INSTALL_DIR}/ --target=mipsel-linux > ${BUILD_DIR}/,gnu-nl-config.log 2>&1
rc=$?
if [ $rc == 0 ]; then
	echo "Configure of g-t-nl succeeded"
else
	echo "CONFIGURE g-t-nl FAILED, $rc"
	exit $rc
fi

gmake -k ALL_TARGET_MODULES= CONFIGURE_TARGET_MODULES= INSTALL_TARGET_MODULES= LANGUAGES="c" > ${BUILD_DIR}/,gnu-nl-build.log 2>&1
rc=$?
if [ $rc == 0 ]; then
	echo "Make of g-t-nl succeeded"
else
	echo "MAKE g-t-nl FAILED, $rc"
	exit $rc
fi

gmake -k INSTALL_TARGET_MODULES= LANGUAGES="c" install > ${BUILD_DIR}/,gnu-nl-install.log 2>&1
rc=$?
if [ $rc == 0 ]; then
	echo "Install g-t-nl succeeded"
else
	echo "INSTALL g-t-nl FAILED, $rc"
	exit $rc
fi

TARGET=mipsel-linux
INCDIR=$INSTALL_DIR/$TARGET/include
CROSS=$INSTALL_DIR/bin/$TARGET-

echo -n "Doing linux includes: "
date

cd ${LINUX}

if [ ! -f .config - ! -f include/linux/version.h ]; then
	echo "LINUX TREE IS NOT CONFIGURED"
	exit 1
fi

# Make include directory in INSTALL_DIR and copy include files
mkdir -p $INCDIR
cd include && tar -cf - asm asm-mips linux | tar -xf - -C $INCDIR
rc=$?
if [ $rc == 0 ]; then
	echo "Linux include dir copy succeeded"
else
	echo "L-I-D COPY FAILED, $rc"
	exit $rc
fi

GLIBC=${BUILD_DIR}/glibc

cd ${GLIBC}

mkdir mipsel-obj
mkdir mipsel-install

cd mipsel-obj

export PATH=${INSTALL_DIR}/bin:$PATH

export CC=${CROSS}gcc
export AS=${CROSS}as
export LD=${CROSS}ld
export AR=${CROSS}ar
export RANLIB=${CROSS}ranlib
export BUILD_CC=gcc

echo -n "Doing glibc: "
date

../configure --prefix= --host=${TARGET} --with-headers=${INCDIR} \
	--enable-kernel=2.3.99 --enable-add-ons --with-elf --disable-profile > ${BUILD_DIR}/,glibc-config.log 2>&1
rc=$?
if [ $rc == 0 ]; then
	echo "Glibc Configure succeeded"
else
	echo "Glibc CONFIGURE FAILED, $rc"
	exit $rc
fi

gmake > ${BUILD_DIR}/,glibc-build.log 2>&1
rc=$?
if [ $rc == 0 ]; then
	echo -n "Glibc Make succeeded"
	date
else
	echo "Glibc MAKE FAILED, $rc"
	exit $rc
fi

gmake install_root=${GLIBC}/mipsel-install install > ${BUILD_DIR}/,glibc-install.log 2>&1
rc=$?
if [ $rc == 0 ]; then
	echo -n "Glibc Install succeeded"
	date
else
	echo "Glibc INSTALL FAILED, $rc"
	exit $rc
fi

cd ../mipsel-install/include && tar -cf - . | tar -xf - -C ${INCDIR}
rc=$?
if [ $rc == 0 ]; then
	echo "Glibc include Install succeeded"
else
	echo "GL-I-I FAILED, $rc"
	exit $rc
fi

cd ../lib && tar -cf - . | tar -xf - -C ${INSTALL_DIR}/${TARGET}/lib
rc=$?
if [ $rc == 0 ]; then
	echo "Glibc lib Install succeeded"
else
	echo "GL-L-I FAILED, $rc"
	exit $rc
fi

echo "Fixing up libc.so locations"
sed -e s%/lib/%"${INSTALL_DIR}/${TARGET}/lib/"%g \
	< "${INSTALL_DIR}/${TARGET}/lib/libc.so" \
	> "${INSTALL_DIR}/${TARGET}/lib/.new.libc.so"
rc=$?
if [ $rc == 0 ]; then
	echo "libc.so Fixup succeeded"
else
	echo "libc.so FIXUP FAILED, $rc"
	exit $rc
fi

mv -f "${INSTALL_DIR}/${TARGET}/lib/.new.libc.so" \
	"${INSTALL_DIR}/${TARGET}/lib/libc.so"
rc=$?
if [ $rc == 0 ]; then
	echo "libc.so Replace succeeded"
else
	echo "libc.so REPLACE FAILED, $rc"
	exit $rc
fi

# Clean up the environment

unset CC
unset AS
unset LD
unset AR
unset RANLIB
unset BUILD_CC
unset CFLAGS
unset ASFLAGS

# Now go back to src/gnu and do a full build & install
cd ${GNU}

mkdir mipsel
cd mipsel

echo -n "Doing full gnu-tools: "
date

../configure -v --enable-shared --enable-threads --enable-languages=c \
	--prefix=${INSTALL_DIR}/ --target=mipsel-linux > ${BUILD_DIR}/,gnu-full-config.log 2>&1
rc=$?
if [ $rc == 0 ]; then
	echo "Configure for g-t-full succeeded"
else
	echo "CONFIGURE g-t-full FAILED, $rc"
	exit $rc
fi

gmake > ${BUILD_DIR}/,gnu-full-build.log 2>&1
rc=$?
if [ $rc == 0 ]; then
	echo "Make for g-t-full succeeded"
else
	echo "MAKE g-t-full FAILED, $rc"
	exit $rc
fi

gmake install > ${BUILD_DIR}/,gnu-full-install.log 2>&1
rc=$?
if [ $rc == 0 ]; then
	echo "Install for g-t-full succeeded"
else
	echo "INSTALL g-t-full FAILED, $rc"
	exit $rc
fi

echo -n "build_tools done: "
date
